/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.core.internal.http.pipeline.stages;

import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.util.Optional;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.checksums.DefaultChecksumAlgorithm;
import software.amazon.awssdk.checksums.spi.ChecksumAlgorithm;
import software.amazon.awssdk.core.ClientType;
import software.amazon.awssdk.core.HttpChecksumConstant;
import software.amazon.awssdk.core.async.AsyncRequestBody;
import software.amazon.awssdk.core.checksums.Algorithm;
import software.amazon.awssdk.core.checksums.ChecksumSpecs;
import software.amazon.awssdk.core.checksums.SdkChecksum;
import software.amazon.awssdk.core.interceptor.ExecutionAttributes;
import software.amazon.awssdk.core.interceptor.InterceptorContext;
import software.amazon.awssdk.core.interceptor.SdkExecutionAttribute;
import software.amazon.awssdk.core.interceptor.SdkInternalExecutionAttribute;
import software.amazon.awssdk.core.internal.async.ChecksumCalculatingAsyncRequestBody;
import software.amazon.awssdk.core.internal.http.RequestExecutionContext;
import software.amazon.awssdk.core.internal.http.pipeline.MutableRequestToRequestPipeline;
import software.amazon.awssdk.core.internal.io.AwsUnsignedChunkedEncodingInputStream;
import software.amazon.awssdk.core.internal.useragent.BusinessMetricsUtils;
import software.amazon.awssdk.core.internal.util.ChunkContentUtils;
import software.amazon.awssdk.core.internal.util.HttpChecksumResolver;
import software.amazon.awssdk.core.internal.util.HttpChecksumUtils;
import software.amazon.awssdk.core.useragent.BusinessMetricCollection;
import software.amazon.awssdk.http.ContentStreamProvider;
import software.amazon.awssdk.http.SdkHttpFullRequest;
import software.amazon.awssdk.http.auth.aws.internal.signer.util.ChecksumUtil;
import software.amazon.awssdk.http.auth.spi.signer.PayloadChecksumStore;
import software.amazon.awssdk.utils.BinaryUtils;
import software.amazon.awssdk.utils.IoUtils;
import software.amazon.awssdk.utils.Md5Utils;

@SdkInternalApi
public class HttpChecksumStage
implements MutableRequestToRequestPipeline {
    private static final ChecksumAlgorithm DEFAULT_ALGORITHM = DefaultChecksumAlgorithm.CRC32;
    private final ClientType clientType;

    public HttpChecksumStage(ClientType clientType) {
        this.clientType = clientType;
    }

    @Override
    public SdkHttpFullRequest.Builder execute(SdkHttpFullRequest.Builder request, RequestExecutionContext context) throws Exception {
        this.ensurePayloadChecksumStorePresent(context.executionAttributes());
        SdkHttpFullRequest.Builder result = HttpChecksumStage.sraSigningEnabled(context) ? this.sraChecksum(request, context) : this.legacyChecksum(request, context);
        this.recordChecksumBusinessMetrics(context.executionAttributes());
        return result;
    }

    private SdkHttpFullRequest.Builder legacyChecksum(SdkHttpFullRequest.Builder request, RequestExecutionContext context) {
        ChecksumSpecs resolvedChecksumSpecs = HttpChecksumResolver.getResolvedChecksumSpecs(context.executionAttributes());
        PayloadChecksumStore checksumStore = this.getPayloadChecksumStore(context.executionAttributes());
        if (this.md5ChecksumRequired(request, context)) {
            this.addMd5ChecksumInHeader(request, checksumStore);
            return request;
        }
        if (this.flexibleChecksumInTrailerRequired(context, resolvedChecksumSpecs)) {
            this.addFlexibleChecksumInTrailer(request, context, resolvedChecksumSpecs);
            return request;
        }
        if (this.flexibleChecksumInHeaderRequired(context, resolvedChecksumSpecs)) {
            this.addFlexibleChecksumInHeader(request, context, resolvedChecksumSpecs, checksumStore);
            return request;
        }
        return request;
    }

    private SdkHttpFullRequest.Builder sraChecksum(SdkHttpFullRequest.Builder request, RequestExecutionContext context) {
        ExecutionAttributes executionAttributes = context.executionAttributes();
        if (!HttpChecksumUtils.isHttpChecksumCalculationNeeded(request, executionAttributes)) {
            return request;
        }
        ChecksumSpecs resolvedChecksumSpecs = executionAttributes.getAttribute(SdkExecutionAttribute.RESOLVED_CHECKSUM_SPECS);
        if ((resolvedChecksumSpecs == null || resolvedChecksumSpecs.algorithmV2() == null) && (resolvedChecksumSpecs = HttpChecksumStage.checksumSpecsWithDefaultAlgorithm(resolvedChecksumSpecs)).requestAlgorithmHeader() != null) {
            request.putHeader(resolvedChecksumSpecs.requestAlgorithmHeader(), DEFAULT_ALGORITHM.algorithmId());
        }
        executionAttributes.putAttribute(SdkExecutionAttribute.RESOLVED_CHECKSUM_SPECS, resolvedChecksumSpecs);
        return request;
    }

    private boolean md5ChecksumRequired(SdkHttpFullRequest.Builder request, RequestExecutionContext context) {
        boolean isHttpChecksumRequired = context.executionAttributes().getAttribute(SdkInternalExecutionAttribute.HTTP_CHECKSUM_REQUIRED) != null || HttpChecksumUtils.isMd5ChecksumRequired(context.executionAttributes());
        boolean requestAlreadyHasMd5 = request.firstMatchingHeader("Content-MD5").isPresent();
        if (!isHttpChecksumRequired || requestAlreadyHasMd5) {
            return false;
        }
        if (context.requestProvider() != null) {
            throw new IllegalArgumentException("This operation requires a content-MD5 checksum, but one cannot be calculated for non-blocking content.");
        }
        return context.executionContext().interceptorContext().requestBody().isPresent();
    }

    private void addMd5ChecksumInHeader(SdkHttpFullRequest.Builder request, PayloadChecksumStore checksumStore) {
        try {
            byte[] payloadMd5 = checksumStore.getChecksumValue(DefaultChecksumAlgorithm.MD5);
            if (payloadMd5 == null) {
                payloadMd5 = Md5Utils.computeMD5Hash(request.contentStreamProvider().newStream());
                checksumStore.putChecksumValue(DefaultChecksumAlgorithm.MD5, payloadMd5);
            }
            request.putHeader("Content-MD5", BinaryUtils.toBase64(payloadMd5));
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private static ChecksumSpecs checksumSpecsWithDefaultAlgorithm(ChecksumSpecs resolvedChecksumSpecs) {
        return (resolvedChecksumSpecs == null ? ChecksumSpecs.builder() : resolvedChecksumSpecs.toBuilder()).algorithmV2(DEFAULT_ALGORITHM).headerName(ChecksumUtil.checksumHeaderName(DEFAULT_ALGORITHM)).build();
    }

    private boolean flexibleChecksumInTrailerRequired(RequestExecutionContext context, ChecksumSpecs checksumSpecs) {
        boolean hasRequestBody = this.hasRequestBody(context);
        boolean isContentStreaming = context.executionContext().interceptorContext().requestBody().map(requestBody -> requestBody.contentStreamProvider() != null).orElse(false);
        return checksumSpecs != null && checksumSpecs.headerName() != null && HttpChecksumUtils.isTrailerBasedChecksumForClientType(context.executionAttributes(), context.executionContext().interceptorContext().httpRequest(), this.clientType, checksumSpecs, hasRequestBody, isContentStreaming);
    }

    private boolean hasRequestBody(RequestExecutionContext context) {
        switch (this.clientType) {
            case ASYNC: {
                return context.executionContext().interceptorContext().asyncRequestBody().isPresent();
            }
            case SYNC: {
                return context.executionContext().interceptorContext().requestBody().isPresent();
            }
        }
        throw new UnsupportedOperationException("Unsupported client type: " + (Object)((Object)this.clientType));
    }

    private static boolean sraSigningEnabled(RequestExecutionContext context) {
        return context.executionAttributes().getAttribute(SdkInternalExecutionAttribute.AUTH_SCHEMES) != null && context.signer() == null;
    }

    private void addFlexibleChecksumInTrailer(SdkHttpFullRequest.Builder request, RequestExecutionContext context, ChecksumSpecs checksumSpecs) {
        long originalContentLength = 0L;
        int chunkSize = 0;
        if (this.clientType == ClientType.SYNC) {
            request.contentStreamProvider(new ChecksumCalculatingStreamProvider(request.contentStreamProvider(), checksumSpecs, this.getPayloadChecksumStore(context.executionAttributes())));
            originalContentLength = context.executionContext().interceptorContext().requestBody().get().optionalContentLength().orElse(0L);
            chunkSize = 131072;
        } else if (this.clientType == ClientType.ASYNC && context.requestProvider() != null) {
            ChecksumCalculatingAsyncRequestBody.Builder checksumBodyBuilder = ChecksumCalculatingAsyncRequestBody.builder().asyncRequestBody(context.requestProvider()).algorithm(checksumSpecs.algorithmV2()).checksumStore(this.getPayloadChecksumStore(context.executionAttributes())).trailerHeader(checksumSpecs.headerName());
            Optional<Long> maybeContentLengthHeader = request.firstMatchingHeader("Content-Length").map(Long::parseLong);
            maybeContentLengthHeader.ifPresent(checksumBodyBuilder::contentLengthHeader);
            context.requestProvider((AsyncRequestBody)checksumBodyBuilder.build());
            originalContentLength = maybeContentLengthHeader.orElseGet(() -> context.executionContext().interceptorContext().asyncRequestBody().flatMap(AsyncRequestBody::contentLength).orElse(0L));
            chunkSize = 131072;
        }
        Algorithm legacyAlgo = checksumSpecs.algorithm();
        long checksumContentLength = ChunkContentUtils.calculateChecksumTrailerLength(legacyAlgo, checksumSpecs.headerName());
        long contentLen = checksumContentLength + ChunkContentUtils.calculateStreamContentLength(originalContentLength, chunkSize);
        request.putHeader("x-amz-trailer", checksumSpecs.headerName()).appendHeader("Content-encoding", "aws-chunked").putHeader("x-amz-content-sha256", "STREAMING-UNSIGNED-PAYLOAD-TRAILER").putHeader("x-amz-decoded-content-length", Long.toString(originalContentLength)).putHeader("Content-Length", Long.toString(contentLen));
    }

    private boolean flexibleChecksumInHeaderRequired(RequestExecutionContext context, ChecksumSpecs headerChecksumSpecs) {
        if (!context.executionContext().interceptorContext().requestBody().isPresent()) {
            return false;
        }
        InterceptorContext interceptorContext = context.executionContext().interceptorContext();
        boolean isContentStreaming = context.executionContext().interceptorContext().requestBody().map(requestBody -> requestBody.contentStreamProvider() != null).orElse(false);
        return headerChecksumSpecs != null && headerChecksumSpecs.algorithm() != null && !HttpChecksumUtils.isHttpChecksumPresent(interceptorContext.httpRequest(), headerChecksumSpecs) && HttpChecksumUtils.isUnsignedPayload(context.executionAttributes().getAttribute(HttpChecksumConstant.SIGNING_METHOD), interceptorContext.httpRequest().protocol(), isContentStreaming) && !headerChecksumSpecs.isRequestStreaming();
    }

    private void addFlexibleChecksumInHeader(SdkHttpFullRequest.Builder request, RequestExecutionContext context, ChecksumSpecs checksumSpecs, PayloadChecksumStore checksumStore) {
        try {
            Algorithm legacyAlgorithm = checksumSpecs.algorithm();
            ChecksumAlgorithm newAlgorithm = HttpChecksumUtils.toNewChecksumAlgorithm(legacyAlgorithm);
            byte[] payloadChecksum = checksumStore.getChecksumValue(newAlgorithm);
            if (payloadChecksum == null) {
                payloadChecksum = HttpChecksumUtils.computeChecksum(context.executionContext().interceptorContext().requestBody().get().contentStreamProvider().newStream(), legacyAlgorithm);
                checksumStore.putChecksumValue(newAlgorithm, payloadChecksum);
            }
            String headerValue = BinaryUtils.toBase64(payloadChecksum);
            request.putHeader(checksumSpecs.headerName(), headerValue);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void ensurePayloadChecksumStorePresent(ExecutionAttributes executionAttributes) {
        PayloadChecksumStore cache = this.getPayloadChecksumStore(executionAttributes);
        if (cache == null) {
            cache = PayloadChecksumStore.create();
            executionAttributes.putAttribute(SdkInternalExecutionAttribute.CHECKSUM_STORE, cache);
        }
    }

    private PayloadChecksumStore getPayloadChecksumStore(ExecutionAttributes executionAttributes) {
        return executionAttributes.getAttribute(SdkInternalExecutionAttribute.CHECKSUM_STORE);
    }

    private void recordChecksumBusinessMetrics(ExecutionAttributes executionAttributes) {
        BusinessMetricCollection businessMetrics = executionAttributes.getAttribute(SdkInternalExecutionAttribute.BUSINESS_METRICS);
        if (businessMetrics == null) {
            return;
        }
        BusinessMetricsUtils.resolveRequestChecksumCalculationMetric(executionAttributes.getAttribute(SdkInternalExecutionAttribute.REQUEST_CHECKSUM_CALCULATION)).ifPresent(businessMetrics::addMetric);
        BusinessMetricsUtils.resolveResponseChecksumValidationMetric(executionAttributes.getAttribute(SdkInternalExecutionAttribute.RESPONSE_CHECKSUM_VALIDATION)).ifPresent(businessMetrics::addMetric);
        ChecksumSpecs checksumSpecs = executionAttributes.getAttribute(SdkExecutionAttribute.RESOLVED_CHECKSUM_SPECS);
        if (checksumSpecs != null && checksumSpecs.algorithmV2() != null) {
            BusinessMetricsUtils.resolveChecksumAlgorithmMetric(checksumSpecs.algorithmV2()).ifPresent(businessMetrics::addMetric);
        }
    }

    static final class ChecksumCalculatingStreamProvider
    implements ContentStreamProvider {
        private final ContentStreamProvider underlyingInputStreamProvider;
        private final String checksumHeaderForTrailer;
        private final ChecksumSpecs checksumSpecs;
        private final PayloadChecksumStore checksumStore;
        private InputStream currentStream;
        private final ChecksumAlgorithm checksumAlgorithm;
        private SdkChecksum sdkChecksum;

        ChecksumCalculatingStreamProvider(ContentStreamProvider underlyingInputStreamProvider, ChecksumSpecs checksumSpecs, PayloadChecksumStore checksumStore) {
            this.underlyingInputStreamProvider = underlyingInputStreamProvider;
            this.sdkChecksum = SdkChecksum.forAlgorithm(checksumSpecs.algorithm());
            this.checksumAlgorithm = HttpChecksumUtils.toNewChecksumAlgorithm(checksumSpecs.algorithm());
            this.checksumHeaderForTrailer = checksumSpecs.headerName();
            this.checksumSpecs = checksumSpecs;
            this.checksumStore = checksumStore;
        }

        @Override
        public InputStream newStream() {
            this.closeCurrentStream();
            this.currentStream = ((AwsUnsignedChunkedEncodingInputStream.Builder)((AwsUnsignedChunkedEncodingInputStream.Builder)((AwsUnsignedChunkedEncodingInputStream.Builder)((AwsUnsignedChunkedEncodingInputStream.Builder)((AwsUnsignedChunkedEncodingInputStream.Builder)AwsUnsignedChunkedEncodingInputStream.builder().inputStream(this.underlyingInputStreamProvider.newStream())).checksumAlgorithm(this.checksumAlgorithm)).sdkChecksum(this.sdkChecksum)).checksumStore(this.checksumStore)).checksumHeaderForTrailer(this.checksumHeaderForTrailer)).build();
            return this.currentStream;
        }

        private void closeCurrentStream() {
            this.sdkChecksum = SdkChecksum.forAlgorithm(this.checksumSpecs.algorithm());
            if (this.currentStream != null) {
                IoUtils.closeQuietly(this.currentStream, null);
                this.currentStream = null;
            }
        }
    }
}

