/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.http.nio.netty.internal.http2;

import io.netty.channel.Channel;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.SimpleChannelInboundHandler;
import io.netty.handler.codec.http2.DefaultHttp2PingFrame;
import io.netty.handler.codec.http2.Http2PingFrame;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.GenericFutureListener;
import io.netty.util.concurrent.ScheduledFuture;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.http.Protocol;
import software.amazon.awssdk.http.nio.netty.internal.ChannelAttributeKey;
import software.amazon.awssdk.http.nio.netty.internal.http2.PingFailedException;
import software.amazon.awssdk.http.nio.netty.internal.utils.NettyClientLogger;
import software.amazon.awssdk.utils.Validate;

@SdkInternalApi
public class Http2PingHandler
extends SimpleChannelInboundHandler<Http2PingFrame> {
    private static final NettyClientLogger log = NettyClientLogger.getLogger(Http2PingHandler.class);
    private static final Http2PingFrame DEFAULT_PING_FRAME = new DefaultHttp2PingFrame(0L);
    private final long delayWarningTimeLimitMs;
    private final long pingTimeoutMillis;
    private ScheduledFuture<?> periodicPing;
    private long lastPingSendTime = 0L;
    private long lastPingAckTime = 0L;

    public Http2PingHandler(int pingTimeoutMillis) {
        this.pingTimeoutMillis = pingTimeoutMillis;
        this.delayWarningTimeLimitMs = Math.min(100, pingTimeoutMillis / 10);
    }

    @Override
    public void handlerAdded(ChannelHandlerContext ctx) {
        CompletableFuture<Protocol> protocolFuture = ctx.channel().attr(ChannelAttributeKey.PROTOCOL_FUTURE).get();
        Validate.validState(protocolFuture != null, "Protocol future must be initialized before handler is added.", new Object[0]);
        protocolFuture.thenAccept(p -> this.start((Protocol)((Object)p), ctx));
    }

    private void start(Protocol protocol, ChannelHandlerContext ctx) {
        if (protocol == Protocol.HTTP2 && this.periodicPing == null) {
            this.periodicPing = ctx.channel().eventLoop().schedule(() -> this.doPeriodicPing(ctx.channel()), 0L, TimeUnit.MILLISECONDS);
        }
    }

    @Override
    public void handlerRemoved(ChannelHandlerContext ctx) {
        this.stop();
    }

    @Override
    public void channelInactive(ChannelHandlerContext ctx) {
        this.stop();
        ctx.fireChannelInactive();
    }

    @Override
    protected void channelRead0(ChannelHandlerContext ctx, Http2PingFrame frame) {
        log.debug(ctx.channel(), () -> "Received PING from channel, ack=" + frame.ack());
        if (frame.ack()) {
            this.lastPingAckTime = System.currentTimeMillis();
        } else {
            ctx.fireChannelRead(frame);
        }
    }

    private void doPeriodicPing(Channel channel) {
        if (this.lastPingAckTime <= this.lastPingSendTime - this.pingTimeoutMillis) {
            log.warn(channel, () -> "PING timeout occurred");
            long timeSinceLastPingSend = System.currentTimeMillis() - this.lastPingSendTime;
            this.channelIsUnhealthy(channel, new PingFailedException("Server did not respond to PING after " + timeSinceLastPingSend + "ms (limit: " + this.pingTimeoutMillis + "ms)"));
        } else {
            long scheduleTime;
            log.debug(channel, () -> "Sending HTTP2/PING frame");
            long l = scheduleTime = this.lastPingSendTime == 0L ? 0L : System.currentTimeMillis() - this.lastPingSendTime;
            if (scheduleTime - this.pingTimeoutMillis > this.delayWarningTimeLimitMs) {
                log.warn(channel, () -> "PING timer scheduled after " + scheduleTime + "ms");
            }
            this.sendPing(channel);
        }
    }

    private void sendPing(Channel channel) {
        long writeMs = System.currentTimeMillis();
        channel.writeAndFlush(DEFAULT_PING_FRAME).addListener((GenericFutureListener<? extends Future<? super Void>>)((GenericFutureListener<Future>)res -> {
            if (!res.isSuccess()) {
                log.debug(channel, () -> "Failed to write and flush PING frame to connection", res.cause());
                this.channelIsUnhealthy(channel, new PingFailedException("Failed to send PING to the service", res.cause()));
            } else {
                log.debug(channel, () -> "Successfully flushed PING frame to connection");
                this.lastPingSendTime = System.currentTimeMillis();
                long flushTime = this.lastPingSendTime - writeMs;
                if (flushTime > this.delayWarningTimeLimitMs) {
                    log.warn(channel, () -> "Flushing PING frame took " + flushTime + "ms");
                }
                this.periodicPing = channel.eventLoop().schedule(() -> this.doPeriodicPing(channel), this.pingTimeoutMillis, TimeUnit.MILLISECONDS);
            }
        }));
    }

    private void channelIsUnhealthy(Channel channel, PingFailedException exception) {
        this.stop();
        channel.pipeline().fireExceptionCaught(exception);
    }

    private void stop() {
        if (this.periodicPing != null) {
            this.periodicPing.cancel(false);
            this.periodicPing = null;
        }
    }
}

